#if PLATFORM_UNIFIED                 
using ObjCRuntime;
[assembly: ObjCRuntime.LinkWith(
        "zumero_client_api.a",
        ForceLoad = true,
        LinkerFlags = "",
        Frameworks = ""
        )
        ]
#endif

namespace Zumero
{
    using System;
    using System.Text;
    using System.Collections.Generic;
    using System.IO;
    using System.Reflection;
    using System.Runtime.InteropServices;
#if !WINDOWS_WINRT
    using System.Runtime.Serialization.Formatters.Binary;
#endif
    using System.Runtime.Serialization;

#if INIT_SQLITEPCL
    using SQLitePCL;
#endif

#if Z_ANDROID
    using Java.Lang;
#endif
#if PLATFORM_IOS
    using MonoTouch;
#elif PLATFORM_UNIFIED
    using ObjCRuntime;
#endif

    /// <summary>
    /// An exception thrown by a Zumero operation.
    /// </summary>
#if !WINDOWS_WINRT
    [Serializable()]
#endif
    internal sealed class ZumeroExceptionInstance : ZumeroException
    {
        private int _rc;
        private string _errstr;
        private string _details;

        public override string ToString()
        {
            return string.Format("{0} ({1}): {2}\r\n{3}", _rc, _errstr, _details, base.ToString());
        }

        public override string Message
        {
            get
            {
                return string.Format("{0}: {1}", _rc, _errstr); ;
            }
        }

        /// <summary>
        /// The <c>ZumeroResult</c> or SQLite result value associated with the exception.
        /// </summary>
        public override int ErrorCode
        {
            get
            {
                return _rc;
            }
        }

        /// <summary>
        /// The errstr associated with the <c>ErrorCode</c> value.
        /// </summary>
        public override string ErrorString
        {
            get
            {
                return _errstr;
            }
        }

        /// <summary>
        /// The detailed error message (if any) returned by the failed Zumero function.
        /// </summary>
        public override string ErrorDetails
        {
            get
            {
                return _details;
            }
        }

        internal ZumeroExceptionInstance(int rc, string errstr, string details)
            : base()
        {
            _rc = rc;
            _errstr = errstr;
            _details = details;
        }
    }
    internal class ZumeroClientImplementation : IZumeroClientImplementation
    {
        static ZumeroClientImplementation()
        {
            Console.Out.WriteLine("ZumeroClientImplementation constructor starting");
#if Z_ANDROID
            Console.Out.WriteLine("(Android) Calling JavaSystem.LoadLibrary");
            Android.App.Application.SynchronizationContext.Send(c => { JavaSystem.LoadLibrary("zumero_sqlite_functions");  JavaSystem.LoadLibrary("zumero_client_api"); }, null);
            Console.Out.WriteLine("(Android) Called JavaSystem.LoadLibrary");
#else
            if (RuntimeInformation.FrameworkDescription.StartsWith(".NET Framework"))
            {
                Console.Out.WriteLine("Calling TryLoadFromArchDirectory");
                TryLoadFromArchDirectory();
                Console.Out.WriteLine("Called TryLoadFromArchDirectory");
            }
#endif
#if INIT_SQLITEPCL
            new SQLitePCL.SQLite3Provider_e_sqlite3();
#endif
#if INIT_SQLITEPCL_SQLCIPHER
            new SQLitePCL.SQLite3Provider_sqlcipher();
#endif
        }

	// TODO on which Windows platforms is LoadLibraryEx available?

        [DllImport("kernel32")]
        private static extern IntPtr LoadLibraryEx(string lpFileName, IntPtr hFile, uint dwFlags);

        // Copyright (c) Microsoft Open Technologies, Inc. All rights reserved.
        // Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.
        // https://github.com/aspnet/DataCommon.SQLite/blob/dev/src/Microsoft.Data.SQLite/Utilities/NativeLibraryLoader.cs


        private static bool TryLoadFromArchDirectory()
        {
            Console.Out.WriteLine("Inside TryLoadFromArchDirectory");
            var dllName = MY_DLL;

		    
#if OLD_REFLECTION
			var currentAssembly = typeof(ZumeroClientImplementation).Assembly;
#else
			var currentAssembly = typeof(ZumeroClientImplementation).GetTypeInfo().Assembly;
#endif
            Console.Out.WriteLine("Got currentAssembly");
            var codeBase = currentAssembly.CodeBase;
			var uri = new UriBuilder(codeBase);
			string baseDirectory = System.IO.Path.GetDirectoryName(Uri.UnescapeDataString(uri.Path));
            Console.Out.WriteLine("Called GetDirectoryName");

            if (baseDirectory == null || !System.IO.Directory.Exists(baseDirectory))
		    {
                baseDirectory = new Uri(AppDomain.CurrentDomain.BaseDirectory).LocalPath;
                Console.Out.WriteLine("Called LocalPath");
            }
            if (baseDirectory == null)
            {
                return false;
            }
            System.Diagnostics.Debug.Assert(System.IO.Path.IsPathRooted(baseDirectory), "baseDirectory is not rooted.");

            var architecture = IntPtr.Size == 4
                ? "x86"
                : "x64";

            var dllPath = System.IO.Path.Combine(
                System.IO.Path.Combine(
                                        System.IO.Path.Combine(
                                            System.IO.Path.Combine(baseDirectory, "runtimes")
                                            , System.String.Format("win-{0}", architecture))
                                        , "native"), dllName);
            Console.Out.WriteLine("loading from " + dllPath);
            if (!System.IO.File.Exists(dllPath))
    	    {
                Console.Out.WriteLine("TryLoadFromArchDirectory " + dllPath + " does not exist");
                return false;
	        }

	        const uint LOAD_WITH_ALTERED_SEARCH_PATH = 8;
            Console.Out.WriteLine("Calling  LoadLibraryEx");
            var ptr = LoadLibraryEx(dllPath, IntPtr.Zero, LOAD_WITH_ALTERED_SEARCH_PATH);
            Console.Out.WriteLine("Called  LoadLibraryEx. Returning from TryLoadFromArchDirectory");
            return ptr != IntPtr.Zero;
        }
        internal static class utils
        {
            internal static byte[] to_utf8(UTF8Encoding enc, string sourceText)
            {
                if (sourceText == null)
                {
                    return null;
                }

                System.Byte[] byteArray;
                int nlen = enc.GetByteCount(sourceText) + 1;

                byteArray = new byte[nlen];
                nlen = enc.GetBytes(sourceText, 0, sourceText.Length, byteArray, 0);
                byteArray[nlen] = 0;

                return byteArray;
            }

            internal static string from_utf8(UTF8Encoding enc, IntPtr nativestring, int nativestringlen)
            {
                if (IntPtr.Zero == nativestring)
                {
                    return null;
                }

                if (0 == nativestringlen)
                {
                    return "";
                }

                if (-1 == nativestringlen)
                {
                    do
                    {
                        nativestringlen++;
                    } while (Marshal.ReadByte(nativestring, nativestringlen) != 0);
                }

                byte[] byteArray = new byte[nativestringlen];

                Marshal.Copy(nativestring, byteArray, 0, nativestringlen);

                return enc.GetString(byteArray, 0, nativestringlen);
            }

        }
        // ----------------------------------------------------------------

#if Z_DLLIMPORT_INTERNAL
        private const string MY_DLL = "__Internal";
#elif Z_DLLIMPORT_ZUMERO_CLIENT_API
        private const string MY_DLL = "zumero_client_api";
#elif Z_DLLIMPORT_ZUMERO_DLL
        private const string MY_DLL = "zumero_client_api.dll";
#endif

        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern void zumero_cancel(int cancellationToken);

        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern int zumero_sync(
                byte[] zFilename,
                byte[] zCipherKey,
                byte[] zServerUrl,
                byte[] zDbFile,
                byte[] zAuthScheme,
                byte[] zUser,
                byte[] zPassword,
                byte[] zTempDir,
                out IntPtr pszErrorDetails
                );

        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern int zumero_sync2(
                byte[] zFilename,
                byte[] zCipherKey,
                byte[] zServerUrl,
                byte[] zDbFile,
                byte[] zAuthScheme,
                byte[] zUser,
                byte[] zPassword,
                byte[] zTempDir,
                ZumeroClientImplementation.callback_progress_handler progress,
                IntPtr dataobject,
                out IntPtr pszErrorDetails
                );
        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern int zumero_sync3(
                byte[] zFilename,
                byte[] zCipherKey,
                byte[] zServerUrl,
                byte[] zDbFile,
                byte[] zAuthScheme,
                byte[] zUser,
                byte[] zPassword,
                byte[] zTempDir,
                ZumeroClientImplementation.callback_progress_handler progress,
                IntPtr dataobject,
                byte[] zJsSyncOptions,
                ref Int32 syncId,
                out IntPtr pszErrorDetails
                );

        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern int zumero_sync_quarantine(
                byte[] zFilename,
                byte[] zCipherKey,
                long qid,
                byte[] zServerUrl,
                byte[] zDbFile,
                byte[] zAuthScheme,
                byte[] zUser,
                byte[] zPassword,
                byte[] zTempDir,
                out IntPtr pszErrorDetails
                );


        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern int zumero_sync_quarantine3(
                byte[] zFilename,
                byte[] zCipherKey,
                long qid,
                byte[] zServerUrl,
                byte[] zDbFile,
                byte[] zAuthScheme,
                byte[] zUser,
                byte[] zPassword,
                byte[] zTempDir,
                ZumeroClientImplementation.callback_progress_handler progress,
                IntPtr dataobject,
                byte[] zJsSyncOptions,
                ref Int32 syncId,
                out IntPtr pszErrorDetails
                );

        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern int zumero_quarantine_since_last_sync(
                byte[] zFilename,
                byte[] zCipherKey,
                byte[] zTempDir,
                ref long pi_qid,
                out IntPtr pszErrorDetails
                );

        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern int zumero_quarantine_since_last_sync3(
                byte[] zFilename,
                byte[] zCipherKey,
                byte[] zTempDir,
                byte[] zJsOptions,
                ref long pi_qid,
                out IntPtr pszErrorDetails
                );

        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern int zumero_delete_quarantine(
                byte[] zFilename,
                byte[] zCipherKey,
                long qid,
                out IntPtr pszErrorDetails
                );

        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern int zumero_delete_quarantine3(
                byte[] zFilename,
                byte[] zCipherKey,
                long qid,
                byte[] zJsOptions,
                out IntPtr pszErrorDetails
                );

        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern IntPtr zumero_errstr(int rc);

        [DllImport(MY_DLL, CallingConvention = CallingConvention.Cdecl)]
        private static extern void zumero_free(IntPtr p);



        // ----------------------------------------------------------------

        /// <summary>
        /// Wrapper around zumero_sync.
        /// See its docs for usage info, and see the remarks on
        /// <see cref="ZumeroClient" /> for general wrapper information.
        /// </summary>
        /// <param name='fileName'>Path of the SQLite database file to operate on.</param>
        /// <param name='cipherKey'>Key to use to decrypt the SQLite database.</param>
        /// <param name='serverUrl'>URL of the server to connect to.</param>
        /// <param name='dbfile'>Name of the dbfile on the server to operate on.</param>
        /// <param name='authScheme'>Credentials for this operation: the authentication scheme to use.</param>
        /// <param name='user'>Credentials for this operation: username to authenticate as.</param>
        /// <param name='password'>Credentials for this operation: password for the specified username.</param>
        public void Sync(
                string fileName,
                string cipherKey,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password
                )
        {
            Sync(fileName, cipherKey, serverUrl, dbfile, authScheme, user, password, null);
        }

        
#if PLATFORM_IOS || PLATFORM_UNIFIED
        [MonoPInvokeCallback (typeof(callback_progress_handler))] // TODO not xplat
#endif
        static private void progress_handler_hook_bridge_impl(int cancellationToken, int phase, UInt64 bytesSoFar, UInt64 bytesTotal, IntPtr p)
        {
            progress_handler_hook_info hi = progress_handler_hook_info.from_ptr(p);
            if (hi != null)
                hi.call(cancellationToken, phase, bytesSoFar, bytesTotal);
        }

        [UnmanagedFunctionPointerAttribute(CallingConvention.Cdecl)]
        private delegate void callback_progress_handler(int cancellationToken, int phase, UInt64 bytesSoFar, UInt64 bytesTotal, IntPtr p);

        callback_progress_handler progress_handler_hook_bridge = new callback_progress_handler(progress_handler_hook_bridge_impl);
        /// <summary>
        /// Wrapper around zumero_sync.
        /// See its docs for usage info, and see the remarks on
        /// <see cref="ZumeroClient" /> for general wrapper information.
        /// </summary>
        /// <param name='fileName'>Path of the SQLite database file to operate on.</param>
        /// <param name='cipherKey'>Key to use to decrypt the SQLite database.</param>
        /// <param name='serverUrl'>URL of the server to connect to.</param>
        /// <param name='dbfile'>Name of the dbfile on the server to operate on.</param>
        /// <param name='authScheme'>Credentials for this operation: the authentication scheme to use.</param>
        /// <param name='user'>Credentials for this operation: username to authenticate as.</param>
        /// <param name='password'>Credentials for this operation: password for the specified username.</param>
        /// <param name='callback'>A callback function that will recieve progress and cancellation information.</param>
        public void Sync(
                string fileName,
                string cipherKey,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password,
                ZumeroClient.callback_progress_handler callback
            )
        {
            int syncid = 0;

            Sync(fileName, cipherKey, serverUrl, dbfile, authScheme, user, password, "", out syncid, callback);
        }

        public void Sync(
                string fileName,
                string cipherKey,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password,
                string jsOptions,
                out Int32 pSyncId,
                ZumeroClient.callback_progress_handler callback
            )
        {
            IntPtr utf8_error_details = IntPtr.Zero;
            UTF8Encoding utf8 = new UTF8Encoding();

            int syncId = -1;

            int rc = 0;

            progress_handler_hook_info progress_hook = null;
            
            if (callback != null)
            {
                progress_hook = new progress_handler_hook_info(callback, null);
            }
            Console.Out.WriteLine("Calling Zumero Sync.");
            try
            {
                rc = zumero_sync3(
                    utils.to_utf8(utf8, fileName),
                    utils.to_utf8(utf8, cipherKey),
                    utils.to_utf8(utf8, serverUrl),
                    utils.to_utf8(utf8, dbfile),
                    utils.to_utf8(utf8, authScheme),
                    utils.to_utf8(utf8, user),
                    utils.to_utf8(utf8, password),
#if WINDOWS_WINRT
                utils.to_utf8(utf8, Windows.Storage.ApplicationData.Current.TemporaryFolder.Path), 
#else
 utils.to_utf8(utf8, System.IO.Path.GetTempPath()),
#endif

     callback == null ? null : progress_handler_hook_bridge,
                    callback == null ? IntPtr.Zero : progress_hook.ptr,
                    jsOptions == null ? null : utils.to_utf8(utf8, jsOptions),
                    ref syncId,
                    out utf8_error_details
                    );
            }
            catch (DllNotFoundException ex)
            {
                throw new DllNotFoundException("Have you included the Sourcegear.sqlite3 nuget package? If you need a version of SQLite that supports encryption, contact support@zumero.com. See https://sqlite.sourcegear.com/", ex);
            }
            Console.Out.WriteLine("Called Zumero Sync.");
            pSyncId = syncId;
            string error_details = null;
            if (IntPtr.Zero != utf8_error_details)
            {
                error_details = utils.from_utf8(utf8, utf8_error_details, -1);
                zumero_free(utf8_error_details);
            }
            
            if (progress_hook != null)
            {
                // TODO maybe turn off the hook here, for now
                progress_hook.free();
                progress_hook = null;
            }

            if (rc != 0)
            {
                throw new ZumeroExceptionInstance(rc, utils.from_utf8(new UTF8Encoding(), zumero_errstr(rc), -1), error_details);
            }
        }

        /// <summary>
        /// Wrapper around zumero_sync_quarantine.
        /// See its docs for usage info, and see the remarks on
        /// <see cref="ZumeroClient" /> for general wrapper information.
        /// </summary>
        /// <param name='fileName'>Path of the SQLite database file to operate on.</param>
        /// <param name='cipherKey'>Key to use to decrypt the SQLite database.</param>
        /// <param name="qid">ID of the quarantined package to include.</param>
        /// <param name='serverUrl'>URL of the server to connect to.</param>
        /// <param name='dbfile'>Name of the dbfile on the server to operate on.</param>
        /// <param name='authScheme'>Credentials for this operation: the authentication scheme to use.</param>
        /// <param name='user'>Credentials for this operation: username to authenticate as.</param>
        /// <param name='password'>Credentials for this operation: password for the specified username.</param>
        /// <param name='jsOptions'>A JSON string with additional options.</param>
        /// <param name="pSyncId">Output parameter containing the id of the completed sync</param>
        /// <param name="callback">The progress callback function</param>
        /// <returns>
        /// True if the function can be called again to do additional work, or
        /// false if all possible work is completed. A true result indicates
        /// that the underlying function returned ZUMERO_PARTIAL, while a false
        /// result indicates 0/success.
        /// </returns>
        public bool SyncQuarantine(
                string fileName,
                string cipherKey,
                long qid,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password,
                string jsOptions,
                out Int32 pSyncId,
                ZumeroClient.callback_progress_handler callback
                )
        {
            IntPtr utf8_error_details = IntPtr.Zero;
            UTF8Encoding utf8 = new UTF8Encoding();

            int syncId = -1;
            int rc = 0;

            progress_handler_hook_info progress_hook = null;

            if (callback != null)
            {
                progress_hook = new progress_handler_hook_info(callback, null);
            }

            rc = zumero_sync_quarantine3(
                utils.to_utf8(utf8, fileName),
                utils.to_utf8(utf8, cipherKey),
                qid,
                utils.to_utf8(utf8, serverUrl),
                utils.to_utf8(utf8, dbfile),
                utils.to_utf8(utf8, authScheme),
                utils.to_utf8(utf8, user),
                utils.to_utf8(utf8, password),
#if WINDOWS_WINRT
				utils.to_utf8(utf8, Windows.Storage.ApplicationData.Current.TemporaryFolder.Path),
#else
 utils.to_utf8(utf8, System.IO.Path.GetTempPath()),
#endif
                callback == null ? null : progress_handler_hook_bridge,
                callback == null ? IntPtr.Zero : progress_hook.ptr,
                jsOptions == null ? null : utils.to_utf8(utf8, jsOptions),
                ref syncId,
                out utf8_error_details

                );
            pSyncId = syncId;

            if (progress_hook != null)
            {
                // TODO maybe turn off the hook here, for now
                progress_hook.free();
                progress_hook = null;
            }

            string error_details = null;
            if (IntPtr.Zero != utf8_error_details)
            {
                error_details = utils.from_utf8(utf8, utf8_error_details, -1);
                zumero_free(utf8_error_details);
            }
            if (
                    (rc != 0)
                    && (rc != 201)
               )
            {
                throw new ZumeroExceptionInstance(rc, utils.from_utf8(new UTF8Encoding(), zumero_errstr(rc), -1), error_details);
            }

            return rc == 201;
        }

        public bool SyncQuarantine(
                string fileName,
                string cipherKey,
                long qid,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password
                )
        {
            int syncid = 0;
            return SyncQuarantine(fileName,
                cipherKey,
                qid,
                serverUrl,
                dbfile,
                authScheme,
                user,
                password, null, out syncid, null);
        }

        /// <summary>
        /// Wrapper around zumero_quarantine_since_last_sync.
        /// See its docs for usage info, and see the remarks on
        /// <see cref="ZumeroClient" /> for general wrapper information.
        /// </summary>
        /// <param name='fileName'>Path of the SQLite database file to operate on.</param>
        /// <param name='cipherKey'>Key to use to decrypt the SQLite database.</param>
        /// <param name='jsOptions'>A JSON string with additional options.</param>
        /// <returns>The qid of the new quarantined package.</returns>
        public long QuarantineSinceLastSync(
                string fileName,
                string cipherKey,
                string jsOptions
                )
        {
            IntPtr utf8_error_details = IntPtr.Zero;
            UTF8Encoding utf8 = new UTF8Encoding();

            int rc = 0;
            long qid = 0;
            rc = zumero_quarantine_since_last_sync3(
                utils.to_utf8(utf8, fileName),
                utils.to_utf8(utf8, cipherKey),
#if WINDOWS_WINRT
				utils.to_utf8(utf8, Windows.Storage.ApplicationData.Current.TemporaryFolder.Path),
#else
 utils.to_utf8(utf8, System.IO.Path.GetTempPath()),
#endif
                utils.to_utf8(utf8, jsOptions),
                ref qid,
                out utf8_error_details
                );
            string error_details = null;
            if (IntPtr.Zero != utf8_error_details)
            {
                error_details = utils.from_utf8(utf8, utf8_error_details, -1);
                zumero_free(utf8_error_details);
            }
            if (rc != 0)
            {
                throw new ZumeroExceptionInstance(rc, utils.from_utf8(new UTF8Encoding(), zumero_errstr(rc), -1), error_details);
            }
            return qid;
        }
        public long QuarantineSinceLastSync(
                string fileName,
                string cipherKey
                )
        {
            return QuarantineSinceLastSync(fileName, cipherKey);
        }

        /// <summary>
        /// Wrapper around zumero_delete_quarantine.
        /// See its docs for usage info, and see the remarks on
        /// <see cref="ZumeroClient" /> for general wrapper information.
        /// </summary>
        /// <param name='fileName'>Path of the SQLite database file to operate on.</param>
        /// <param name='cipherKey'>Key to use to decrypt the SQLite database.</param>
        /// <param name='qid'>ID of the quarantined package to delete.</param>
        /// <param name='jsOptions'>A JSON string for additional options.</param>
        public void DeleteQuarantine(
                string fileName,
                string cipherKey,
                long qid,
                string jsOptions
                )
        {
            IntPtr utf8_error_details = IntPtr.Zero;
            UTF8Encoding utf8 = new UTF8Encoding();

            int rc = 0;
            rc = zumero_delete_quarantine3(
                utils.to_utf8(utf8, fileName),
                utils.to_utf8(utf8, cipherKey),
                qid,
                utils.to_utf8(utf8, jsOptions),
                out utf8_error_details
                );
            string error_details = null;
            if (IntPtr.Zero != utf8_error_details)
            {
                error_details = utils.from_utf8(utf8, utf8_error_details, -1);
                zumero_free(utf8_error_details);
            }
            if (rc != 0)
            {
                throw new ZumeroExceptionInstance(rc, utils.from_utf8(new UTF8Encoding(), zumero_errstr(rc), -1), error_details);
            }
        }
        public void DeleteQuarantine(
                string fileName,
                string cipherKey,
                long qid
                )
        {
            DeleteQuarantine(fileName, cipherKey, qid, null);
        }

        public void Cancel(
                int cancellationToken
                )
        {
            zumero_cancel(cancellationToken);
        }
    };

}




