
namespace Zumero
{
    using System;
    using System.Text;
    using System.Collections.Generic;
    using System.IO;
	using System.Runtime.InteropServices;
    
    /// <summary>
    /// Enumeration of possible result codes from Zumero operations.
    /// </summary>
    /// <remarks>
    /// <para>
    /// If a Zumero call throws an exception, then the <c>ZumeroException</c>'s
    /// ErrorCode value will be either one of these codes or one of the
    /// standard SQLite return codes.
    /// </para>
    /// <para>
    /// Each entry in this enumeration has the same numeric value as the
    /// corresponding error code in the Zumero Static Library.
    /// </para>
    /// </remarks>
    public enum ZumeroResult
    {
        Error   = 200,
        Partial = 201,

        NetworkConnectionFailed       = Error | 1<<8,
        AuthenticationFailed          = Error | 2<<8,
        PermissionDenied              = Error | 3<<8,
        Http400                       = Error | 4<<8,
        Http406                       = Error | 5<<8,
        Http500                       = Error | 6<<8,
        Http404                       = Error | 7<<8,
        Http502                       = Error | 8<<8,
        HttpOther                     = Error | 9<<8,
        ConstraintViolation           = Error | 10<<8,
        UniqueConstraintViolation     = Error | 11<<8,
        CheckConstraintViolation      = Error | 12<<8,
        ForeignKeyConstraintViolation = Error | 13<<8,
        SyncRejectByRule              = Error | 14<<8,
        WrongDbfile                   = Error | 15<<8,
        ServerRollbackDetected        = Error | 16<<8,
        DbfileNotFound                = Error | 17<<8,
        TooManyDeadlocks              = Error | 18<<8,
        LicenseError                  = Error | 19<<8,
        InvalidArg                    = Error | 30<<8,
        InvalidDbfileName             = Error | 50<<8,
        InvalidAuthSchemeString       = Error | 51<<8,
        Unsupported                   = Error | 52<<8,
        DifferentCredentials          = Error | 53<<8,
        Cancelled                     = Error | 54<<8,
    }

    /// <summary>
    /// Enumeration of possible phases used in the progress callbacks during a Zumero sync operation.
    /// </summary>
    /// <remarks>
    /// <para>
    /// Each entry in this enumeration has the same numeric value as the
    /// corresponding error code in the Zumero Static Library.
    /// </para>
    /// </remarks>
    public enum ZumeroPhase
    {
        Preparing = 1,
        Uploading = 2,
        WaitingForResponse = 3,
        Downloading = 4,
        Applying = 5,
    }
    public class ZumeroException : Exception
    {
        internal
        ZumeroException()
            : base()
        {

        }

        internal
        ZumeroException(string message)
            : base(message)
        {
    
        }
        public virtual int ErrorCode { get { throw new NotImplementedException("This property should be overridden in the implementation.");  } }
        public virtual string ErrorString { get { throw new NotImplementedException("This property should be overridden in the implementation."); } }
        public virtual string ErrorDetails { get { throw new NotImplementedException("This property should be overridden in the implementation."); } }
    }

    /// <summary>
    /// A static container class for wrappers around Zumero Static Library
    /// functions.
    /// </summary>
    /// <remarks>
    /// <para>
    /// This class is simply a container for static functions that wrap the
    /// functions in the Zumero Static Library.
    /// </para>
    /// <para>
    /// All wrapper functions handle error results from the underlying function
    /// by throwing a ZumeroException. The error details returned from the
    /// function are available from its ErrorDetails property.
    /// </para>
    /// <para>
    /// Passing the standard .NET null value to a wrapper function for unused
    /// parameters will be correctly translated to the C NULL value when
    /// calling the underlying wrapped function. Note that not all parameters
    /// support NULL; consult the C level API documentation for more info.
    /// </para>
    /// </remarks>
    public static class ZumeroClient
    {
        private static ZumeroClientImplementation zci = new ZumeroClientImplementation();
	
        /// <summary>
        /// Wrapper around zumero_sync.
        /// See its docs for usage info, and see the remarks on
        /// <see cref="ZumeroClient" /> for general wrapper information.
        /// </summary>
        /// <param name='fileName'>Path of the SQLite database file to operate on.</param>
        /// <param name='cipherKey'>Key to use to decrypt the SQLite database.</param>
        /// <param name='serverUrl'>URL of the server to connect to.</param>
        /// <param name='dbfile'>Name of the dbfile on the server to operate on.</param>
        /// <param name='authScheme'>Credentials for this operation: the authentication scheme to use.</param>
        /// <param name='user'>Credentials for this operation: username to authenticate as.</param>
        /// <param name='password'>Credentials for this operation: password for the specified username.</param>
        public static void Sync(
                string fileName,
                string cipherKey,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password
                )
	    {
            zci.Sync(fileName, cipherKey, serverUrl, dbfile, authScheme, user, password);
	    }

        /// <summary>
        /// Wrapper around zumero_sync2.
        /// See its docs for usage info, and see the remarks on
        /// <see cref="ZumeroClient" /> for general wrapper information.
        /// </summary>
        /// <param name='fileName'>Path of the SQLite database file to operate on.</param>
        /// <param name='cipherKey'>Key to use to decrypt the SQLite database.</param>
        /// <param name='serverUrl'>URL of the server to connect to.</param>
        /// <param name='dbfile'>Name of the dbfile on the server to operate on.</param>
        /// <param name='authScheme'>Credentials for this operation: the authentication scheme to use.</param>
        /// <param name='user'>Credentials for this operation: username to authenticate as.</param>
        /// <param name='password'>Credentials for this operation: password for the specified username.</param>
        /// <param name='callback'>A callback function that will recieve progress and cancellation information.</param>
        public static void Sync(
                string fileName,
                string cipherKey,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password,
                ZumeroClient.callback_progress_handler callback
            )
        {
            zci.Sync(fileName, cipherKey, serverUrl, dbfile, authScheme, user, password, callback);
        }

		public static void Sync(
				string fileName,
				string cipherKey,
				string serverUrl,
				string dbfile,
				string authScheme,
				string user,
				string password,
				string jsOptions,
				out int syncId,
				ZumeroClient.callback_progress_handler callback
			)
		{
			zci.Sync(fileName, cipherKey, serverUrl, dbfile, authScheme, user, password, jsOptions, out syncId, callback);
		}

        /// <summary>
        /// Wrapper around zumero_sync_quarantine.
        /// See its docs for usage info, and see the remarks on
        /// <see cref="ZumeroClient" /> for general wrapper information.
        /// </summary>
        /// <param name='fileName'>Path of the SQLite database file to operate on.</param>
        /// <param name='cipherKey'>Key to use to decrypt the SQLite database.</param>
        /// <param name="qid">ID of the quarantined package to include.</param>
        /// <param name='serverUrl'>URL of the server to connect to.</param>
        /// <param name='dbfile'>Name of the dbfile on the server to operate on.</param>
        /// <param name='authScheme'>Credentials for this operation: the authentication scheme to use.</param>
        /// <param name='user'>Credentials for this operation: username to authenticate as.</param>
        /// <param name='password'>Credentials for this operation: password for the specified username.</param>
        /// <returns>
        /// True if the function can be called again to do additional work, or
        /// false if all possible work is completed. A true result indicates
        /// that the underlying function returned ZUMERO_PARTIAL, while a false
        /// result indicates 0/success.
        /// </returns>
        public static bool SyncQuarantine(
                string fileName,
                string cipherKey,
                long qid,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password
                )
	    {
            return zci.SyncQuarantine(fileName, cipherKey, qid, serverUrl, dbfile, authScheme, user, password);
	    }

        public static bool SyncQuarantine(
                string fileName,
                string cipherKey,
                long qid,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password,
                string jsOptions,
                out int syncId,
                ZumeroClient.callback_progress_handler callback
                )
        {
            return zci.SyncQuarantine(fileName, cipherKey, qid, serverUrl, dbfile, authScheme, user, password, jsOptions, out syncId, callback);
        }

        /// <summary>
        /// Wrapper around zumero_quarantine_since_last_sync.
        /// See its docs for usage info, and see the remarks on
        /// <see cref="ZumeroClient" /> for general wrapper information.
        /// </summary>
        /// <param name='fileName'>Path of the SQLite database file to operate on.</param>
        /// <param name='cipherKey'>Key to use to decrypt the SQLite database.</param>
        /// <param name="jsOptions">Optional JSON string to provide configuration parameters</param>
        /// <returns>The qid of the new quarantined package.</returns>
        public static long QuarantineSinceLastSync(
                string fileName,
                string cipherKey,
                string jsOptions = null
                )
	    {
            return zci.QuarantineSinceLastSync(fileName, cipherKey, jsOptions);
	    }

        /// <summary>
        /// Wrapper around zumero_delete_quarantine.
        /// See its docs for usage info, and see the remarks on
        /// <see cref="ZumeroClient" /> for general wrapper information.
        /// </summary>
        /// <param name='fileName'>Path of the SQLite database file to operate on.</param>
        /// <param name='cipherKey'>Key to use to decrypt the SQLite database.</param>
        /// <param name='qid'>ID of the quarantined package to delete.</param>
        /// <param name="jsOptions">Optional JSON string to provide configuration parameters</param>
        public static void DeleteQuarantine(
                string fileName,
                string cipherKey,
                long qid,
                string jsOptions = null
                )
	    {
            zci.DeleteQuarantine(fileName, cipherKey, qid, jsOptions);
	    }

        /// <summary>
        /// Wrapper around zumero_cancel
        /// See its docs for usage info, and see the remarks on
        /// <see cref="ZumeroClient" /> for general wrapper information.
        /// </summary>
        /// <param name='cancellationToken'>The cancellation token provided by the Zumero callback.</param>
        public static void Cancel(
                int cancellationToken
                )
        {
            zci.Cancel(cancellationToken);
        }
	    
		
		public delegate void callback_progress_handler(int cancellation_token, int phase, UInt64 bytesSoFar, UInt64 bytesTotal);

    }

    internal interface IZumeroClientImplementation
    {
        void Sync(
                string fileName,
                string cipherKey,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password
                );

        void Sync(
                string fileName,
                string cipherKey,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password,
                ZumeroClient.callback_progress_handler callback
            );

		void Sync(
				string fileName,
				string cipherKey,
				string serverUrl,
				string dbfile,
				string authScheme,
				string user,
				string password,
				string jsOptions,
				out int syncId,
				ZumeroClient.callback_progress_handler callback
			);

        bool SyncQuarantine(
                string fileName,
                string cipherKey,
                long qid,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password,
                string jsOptions,
                out int syncId,
                ZumeroClient.callback_progress_handler callback
                );

        bool SyncQuarantine(
                string fileName,
                string cipherKey,
                long qid,
                string serverUrl,
                string dbfile,
                string authScheme,
                string user,
                string password
                );

        long QuarantineSinceLastSync(
                string fileName,
                string cipherKey,
                string jsOptions
                );

        long QuarantineSinceLastSync(
                string fileName,
                string cipherKey
                );

        void DeleteQuarantine(
                string fileName,
                string cipherKey,
                long qid,
                string jsOptions
                );

        void DeleteQuarantine(
                string fileName,
                string cipherKey,
                long qid
                );

        void Cancel(
                int cancellationToken
                );

    };

    internal class progress_handler_hook_info
    {
        private ZumeroClient.callback_progress_handler _func;
        private object _user_data;
#if ! Z_BAIT
        private GCHandle _h;
#endif
        internal progress_handler_hook_info(ZumeroClient.callback_progress_handler func, object v)
        {
            _func = func;
            _user_data = v;
#if ! Z_BAIT
            _h = GCHandle.Alloc(this);
#endif
        }

        internal IntPtr ptr
        {
            get
            {
#if ! Z_BAIT
                return (IntPtr)_h;
#else
                return IntPtr.Zero;
#endif
            }
        }

        internal static progress_handler_hook_info from_ptr(IntPtr p)
        {
#if ! Z_BAIT
            GCHandle h = (GCHandle)p;
            progress_handler_hook_info hi = h.Target as progress_handler_hook_info;
            // TODO assert(hi._h == h)
            return hi;
#else
            return null;
#endif
        }

        internal void call(int cancel_token, int phase, UInt64 bytesSoFar, UInt64 bytesTotal)
        {
            if (_func != null)
                _func(cancel_token, phase, bytesSoFar, bytesTotal);
        }

        internal void free()
        {
            _func = null;
            _user_data = null;
#if ! Z_BAIT
            _h.Free();
#endif
        }
    };
    
}


