package com.zumero.internal;

import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.database.Cursor;
import java.util.ArrayList;
import java.nio.ByteBuffer;
import android.util.Base64;
import android.util.Log;

import com.zumero.internal.ZumeroLongHolder;
import com.zumero.internal.ZumeroStringHolder;

public class ZumeroStatement {
	public ZumeroDatabase _db = null;
	public String _sql = null;
	public SQLiteStatement _stmt = null;
	public Cursor _cursor = null;
	public boolean _haveAlreadyStepped = false;
	public ArrayList<String> _argsArray = new ArrayList<String>();

	public ZumeroStatement(ZumeroDatabase db, String sql) {
		_db = db;
		_sql = sql;
		//Log.v("ZUMERO", "preparing " + sql);
		if (! _sql.toLowerCase().startsWith("select")
		    && ! _sql.toLowerCase().startsWith("pragma"))
			_stmt = _db._db.compileStatement(sql);
	}

	public static int reset(ZumeroStatement stmt) throws Exception {
		try {
			if (stmt._cursor != null) {
				stmt._cursor.close();
				stmt._cursor = null;
			}
			if (stmt._stmt != null)
				stmt._stmt.clearBindings();
			stmt._argsArray.clear();
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}
	public static int step_for_long(ZumeroStatement stmt, ZumeroLongHolder outputHolder) {
		try {
			outputHolder.val = stmt._stmt.simpleQueryForLong();
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}
	public static int step_for_text(ZumeroStatement stmt, ZumeroStringHolder outputHolder) {
		try {
			outputHolder.val = stmt._stmt.simpleQueryForString();
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}
	public static void dumpDatabase(String tag, ZumeroStatement stmt)
	{
		Log.v(tag, "------------------------------- ");
		Log.v(tag, stmt._sql);
		Cursor c1 = stmt._db._db.rawQuery("SELECT name, type, sql FROM sqlite_master", new String[] {});
		while (c1.moveToNext())
		{
			String type = c1.getString(1);
			Log.v(tag, "==== " + type + " " + c1.getString(0));
			if (!type.contains("table") && !type.contains("view"))
				continue;
			String tableName = c1.getString(0);
			Cursor c2 = stmt._db._db.rawQuery("SELECT * FROM " + tableName, new String[] {});
			while (c2.moveToNext())
			{
				int colCount = c2.getColumnCount();
				for (int i = 0; i < colCount; i++)
				{
					Log.v(tag, type + " " + tableName + "column " + c2.getColumnName(i) + " value " + (c2.getType(i) == Cursor.FIELD_TYPE_BLOB ? "blob" : c2.getString(i)));
				}
			}
			c2.close();
		}
		c1.close();
		c1 = stmt._db._db.rawQuery("SELECT name, type, sql FROM sqlite_temp_master", new String[] {});
		while (c1.moveToNext())
		{
			String type = c1.getString(1);
			if (!type.contains("table") && !type.contains("view"))
				continue;
			Log.v(tag, "==== " + type + " " + c1.getString(0));
			String tableName = c1.getString(0);
			Cursor c2 = stmt._db._db.rawQuery("SELECT * FROM " + tableName, new String[] {});
			while (c2.moveToNext())
			{
				int colCount = c2.getColumnCount();
				for (int i = 0; i < colCount; i++)
				{
					Log.v(tag, type + " " + tableName + " column " + c2.getColumnName(i) + " value " + (c2.isNull(i) ? "FOR REALLY REAL, IT'S NULL" : (c2.getType(i) == Cursor.FIELD_TYPE_BLOB ? "blob" : c2.getString(i))));
				}
			}
			c2.close();
		}
		c1.close();
		Log.v(tag, "------------------------------- ");
	}
	public static int step(ZumeroStatement stmt) throws Exception {
		try {
			//if (stmt._sql.contains("z$zs"))
			{
				//dumpDatabase("BEFORE", stmt);
			}
			int return_code = ZumeroDatabase.SQLITE_ERROR;
			//Log.v("ZUMERO", "step");
			if (stmt._stmt != null) {
				//Log.v("ZUMERO", "database " + stmt._db._fileName + " stepping non-query command " + stmt._sql);
				stmt._stmt.execute();
				return_code = ZumeroDatabase.SQLITE_DONE;
			}
			else {
				//Log.v("ZUMERO", "database " + stmt._db._fileName + " stepping query command " + stmt._sql);
				if (stmt._cursor == null) {
					stmt._cursor = stmt._db._db.rawQuery(stmt._sql, stmt._argsArray.toArray(new String[stmt._argsArray.size()]));
				//Log.v("ZUMERO", "step moving to first for " + stmt._sql);
					if (stmt._cursor.moveToFirst()) {
						//Log.v("ZUMERO", "Cursor has this many columns: " + stmt._cursor.getColumnCount());
						//Log.v("ZUMERO", "Cursor has this many rows: " + stmt._cursor.getCount());
						//Log.v("ZUMERO", "move to first returned true for " + stmt._sql);
						return_code = ZumeroDatabase.SQLITE_ROW;
					}
					else {
						//Log.v("ZUMERO", "move to first returned false for " + stmt._sql);
						return_code = ZumeroDatabase.SQLITE_DONE;
					}
				}
				else {
					//Log.v("ZUMERO", "step moving to next for " + stmt._sql);
					if (stmt._cursor.moveToNext()) {
						//Log.v("ZUMERO", "move to next returned true for " + stmt._sql);
						return_code = ZumeroDatabase.SQLITE_ROW;
					}
					else {
						//Log.v("ZUMERO", "move to next returned false for " + stmt._sql);
						return_code = ZumeroDatabase.SQLITE_DONE;
					}
				}
			}
			//if (stmt._sql.contains("z$zs"))
			{
				//dumpDatabase("AFTER", stmt);
			}
			return return_code;
		}
		catch (Exception e) {
			//Log.v("ZUMERO", "exception stepping SQLite command " + stmt._sql);
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	
	}
	
	public static int bind_int(ZumeroStatement stmt, int colNum, int val) {
		//Log.v("ZUMERO", "bind_int " + stmt._sql + " colNum " + colNum + " val: " + val);
		try {
			if (stmt._stmt != null)
				stmt._stmt.bindLong(colNum, val);
			else {
				growArgsArray(stmt, colNum);
				stmt._argsArray.set(colNum - 1, Long.toString(val));
			}
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}
	
	public static void growArgsArray(ZumeroStatement stmt, int colNum)
	{
		while (colNum > stmt._argsArray.size())
		{
			stmt._argsArray.add(null);
		}
	}

	public static int bind_int64(ZumeroStatement stmt, int colNum, long val) {
		//Log.v("ZUMERO", "bind_int64 " + stmt._sql + " colNum " + colNum + " val: " + val);
		try {
			if (stmt._stmt != null)
				stmt._stmt.bindLong(colNum, val);
			else {
				growArgsArray(stmt, colNum);
				stmt._argsArray.set(colNum - 1, Long.toString(val));
			}
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}

	public static int bind_double(ZumeroStatement stmt, int colNum, double val) {
		//Log.v("ZUMERO", "bind_double " + stmt._sql + " colNum " + colNum + " val: " + val);
		try {
			if (stmt._stmt != null)
				stmt._stmt.bindDouble(colNum, val);
			else {
				growArgsArray(stmt, colNum);
				stmt._argsArray.set(colNum - 1, Double.toString(val));
			}
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}

	public static int bind_null(ZumeroStatement stmt, int colNum) {
		//Log.v("ZUMERO", "bind_null in statement " + stmt._sql + " column " + colNum);
		try {
			if (stmt._stmt != null)
				stmt._stmt.bindNull(colNum);
			else {
				growArgsArray(stmt, colNum);
				stmt._argsArray.set(colNum - 1, null);
			}
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}

	public static int bind_text(ZumeroStatement stmt, int colNum, String val) {
		//Log.v("ZUMERO", "bind_text in statement " + stmt._sql + " column " + colNum + " value " + val);
		try {
			if (stmt._stmt != null)
				stmt._stmt.bindString(colNum, val);
			else {
				growArgsArray(stmt, colNum);
				stmt._argsArray.set(colNum - 1, val);
			}
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}

	public static int bind_zeroblob(ZumeroStatement stmt, int colNum, int num) {
		//Log.v("ZUMERO", "bind_zeroblob in statement " + stmt._sql + " column " + colNum);
		byte[] emptyArray = new byte[num];
		try {
			if (stmt._stmt != null)
				stmt._stmt.bindBlob(colNum, emptyArray);
			else {
				return 1;
			}
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}

	public static int bind_blob(ZumeroStatement stmt, int colNum, byte[] blob) {
				//Log.v("ZUMERO", "for statement " + stmt._sql);
				//if (blob != null)
					//Log.v("ZUMERO", "binding byte[] blob " + colNum + " is not null and has  " + blob.length + " bytes");
				//else
					//Log.v("ZUMERO", "binding null byte[] blob " + colNum + " is null");

		try {
			if (stmt._stmt != null)
			{
				if (blob == null)
					stmt._stmt.bindNull(colNum);
				else
					stmt._stmt.bindBlob(colNum, blob);
			}
			else {
				return 1;
			}
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}

	public static int bind_blob(ZumeroStatement stmt, int colNum, ByteBuffer blob) {
		try {
				//Log.v("ZUMERO", "for statement " + stmt._sql);
				//Log.v("ZUMERO", "binding ByteBuffer blob with length " + blob.limit());

			//Log.v("ZUMERO", "bind_blob is " + blob + " and hasArray() " + blob.hasArray());
			byte[] b = new byte[blob.limit()];
			blob.rewind();
			blob.get(b);
			//for (int i = 0; i < b.length; i++)
			//{
				//Log.v("ZUMERO", "blob[" + i +"] is " + b[i]);
			//}
			//Log.v("ZUMERO", "bind_blob for " + colNum + " length " + b.length);
			if (stmt._stmt != null)
				stmt._stmt.bindBlob(colNum, b);
			else {
				return 1;
			}
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}

	public static int clear_bindings(ZumeroStatement stmt) {
		try {
			if (stmt._stmt != null)
				stmt._stmt.clearBindings();
			stmt._argsArray.clear();
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}

	public static int finalize(ZumeroStatement stmt) {
		//Log.v("ZUMERO", "finalize " + stmt._sql);
		try {
			if (stmt._cursor != null) {
				stmt._cursor.close();
				stmt._cursor = null;
			}
			if (stmt._stmt != null) {
				stmt._stmt.clearBindings();
				stmt._stmt.close();
				stmt._cursor = null;
			}
			return 0;
		}
		catch (Exception e) {
			return ZumeroDatabase.HandleException(stmt._db, e);
		}
	}

	public static int column_bytes(ZumeroStatement stmt, int colNum) {
		try {
			if (column_type(stmt, colNum) == ZumeroDatabase.SQLITE_BLOB)
			{
				byte[] blob = stmt._cursor.getBlob(colNum);
				return blob.length;
			}
			else if (column_type(stmt, colNum) == ZumeroDatabase.SQLITE_TEXT)
			{
				String val = stmt._cursor.getString(colNum);
				return val.length();
			}
			else
				return 0;
		}
		catch (Exception e) {
			ZumeroDatabase.HandleException(stmt._db, e);
		}
		return 0;
	}

	public static ByteBuffer column_blob(ZumeroStatement stmt, int colNum) {
		try {
			if (stmt._cursor != null) {
				byte[] blob = stmt._cursor.getBlob(colNum);
				//Log.v("ZUMERO", "for statement " + stmt._sql);
				//if (blob != null)
					//Log.v("ZUMERO", "blob_column " + colNum + " is not null and has  " + blob.length + " bytes");
				//else
					//Log.v("ZUMERO", "blob_column " + colNum + " is null");
				if (blob != null)
				{
					//if (blob.length > 30)
					//{
						//Log.v("ZUMERO", "for statement " + stmt._sql);
						//Log.v("ZUMERO", "blob_column " + colNum + " is not null and has  " + blob.length + " bytes");
					//}
					//for (int i = 0; i < blob.length; i++)
					//{
						//Log.v("ZUMERO", "blob[" + i +"] is " + blob[i]);
					//}
					ByteBuffer b = ByteBuffer.allocateDirect(blob.length);
					b.put(blob);
					b.rewind();
					//Log.v("ZUMERO", "column_blob ByteBuffer is " + b + " and hasArray() " + b.hasArray() + " and isDirect() " + b.isDirect());
					//Log.v("ZUMERO", "ByteBuffer " + colNum + " is not null and has  " + blob.length + " bytes");

					return b;
				}
				else
				{
					//Log.v("ZUMERO", "blob_column " + colNum + " is null");
					return null;
				}
			}
		}
		catch (Exception e) {
			//Log.v("ZUMERO", "exception getting blob column from command " + stmt._sql);
			ZumeroDatabase.HandleException(stmt._db, e);
		}
		//Log.v("ZUMERO", "blob_column " + colNum + " is returning default null");
		return null;
	}

	public static int column_int(ZumeroStatement stmt, int colNum) {
		try {
			if (stmt._cursor != null) {
				return stmt._cursor.getInt(colNum);
			}
		}
		catch (Exception e) {
			ZumeroDatabase.HandleException(stmt._db, e);
		}
		return 0;
	}

	public static long column_int64(ZumeroStatement stmt, int colNum) {
		try {
			if (stmt._cursor != null) {
				return stmt._cursor.getLong(colNum);
			}
		}
		catch (Exception e) {
			ZumeroDatabase.HandleException(stmt._db, e);
		}
		return 0;
	}

	public static double column_double(ZumeroStatement stmt, int colNum) {
		try {
			if (stmt._cursor != null) {
				return stmt._cursor.getDouble(colNum);
			}
		}
		catch (Exception e) {
			ZumeroDatabase.HandleException(stmt._db, e);
		}
		return 0;
	}

	public static String column_text(ZumeroStatement stmt, int colNum) {
		try {
			if (stmt._cursor != null) {
				//Log.v("ZUMERO", "column_text " + stmt._sql + " value is " + stmt._cursor.getString(colNum));
				return stmt._cursor.getString(colNum);
			}
		}
		catch (Exception e) {
			ZumeroDatabase.HandleException(stmt._db, e);
		}
		//Log.v("ZUMERO", "column_text fall through");
		return null;
	}

	public static String column_name(ZumeroStatement stmt, int colNum) {
		try {
			if (stmt._cursor != null) {
				return stmt._cursor.getColumnName(colNum);
			}
		}
		catch (Exception e) {
			ZumeroDatabase.HandleException(stmt._db, e);
		}
		return null;
	}

	private static void print_value(ZumeroStatement stmt, int colNum)
	{
		int type = stmt._cursor.getType(colNum);
		switch(type) {
			case Cursor.FIELD_TYPE_BLOB:
				//Log.v("ZUMERO", "blob column " + colNum + " has  " + stmt._cursor.getBlob(colNum).length + " bytes");
				return;
			case Cursor.FIELD_TYPE_FLOAT:
				//Log.v("ZUMERO", "float column " + colNum + " is  " + stmt._cursor.getDouble(colNum));
				return;
			case Cursor.FIELD_TYPE_INTEGER:
				//Log.v("ZUMERO", "int column " + colNum + " is  " + stmt._cursor.getLong(colNum));
				return;
			case Cursor.FIELD_TYPE_STRING:
				//Log.v("ZUMERO", "string column " + colNum + " is  " + stmt._cursor.getString(colNum));
				return;
			case Cursor.FIELD_TYPE_NULL:
				//Log.v("ZUMERO", "null column " + colNum + " is null");
				return;
		}
	}
	public static int column_type(ZumeroStatement stmt, int colNum) {
		try {
			if (stmt._cursor != null) {
				int type = stmt._cursor.getType(colNum);
				//Log.v("ZUMERO", "column_type for " + colNum + " is " + type);
				int colCount = stmt._cursor.getColumnCount();
				//for (int i = 0; i < colCount; i++)
				//{
					//print_value(stmt, i);
				//}
				switch(type) {
					case Cursor.FIELD_TYPE_BLOB:
						return ZumeroDatabase.SQLITE_BLOB;
					case Cursor.FIELD_TYPE_FLOAT:
						return ZumeroDatabase.SQLITE_FLOAT;
					case Cursor.FIELD_TYPE_INTEGER:
						return ZumeroDatabase.SQLITE_INTEGER;
					case Cursor.FIELD_TYPE_STRING:
						return ZumeroDatabase.SQLITE_TEXT;
					case Cursor.FIELD_TYPE_NULL:
						return ZumeroDatabase.SQLITE_NULL;
				}
			}
		}
		catch (Exception e) {
			ZumeroDatabase.HandleException(stmt._db, e);
		}
		return -1;
	}

	public static int column_count(ZumeroStatement stmt) {
		try {
			//Log.v("ZUMERO", "java column count for " + stmt._sql);
			if (stmt._cursor != null) {
				//Log.v("ZUMERO", "cursor is not null for " + stmt._sql);
				//Log.v("ZUMERO", "column count is " + stmt._cursor.getColumnCount());
				return stmt._cursor.getColumnCount();
			}
		}
		catch (Exception e) {
			ZumeroDatabase.HandleException(stmt._db, e);
		}
		//Log.v("ZUMERO", "column_count fall through");
		return -1;
	}
}
