package com.zumero.internal;

import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.database.Cursor;
import java.util.ArrayList;
import java.util.Dictionary;
import java.util.Hashtable;
import java.nio.ByteBuffer;
import java.io.FileOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import android.util.Log;
import android.os.ParcelFileDescriptor;
import com.zumero.internal.ZumeroDatabaseHolder;
import com.zumero.internal.ZumeroStatementHolder;
import com.zumero.internal.ZumeroStatement;

public class ZumeroDatabase {
	private static final int SQLITE_OPEN_READONLY         = 0x00000001;  /* Ok for sqlite3_open_v2() */
	private static final int SQLITE_OPEN_READWRITE        = 0x00000002;  /* Ok for sqlite3_open_v2() */
	private static final int SQLITE_OPEN_CREATE           = 0x00000004;  /* Ok for sqlite3_open_v2() */
	private static final int SQLITE_OPEN_DELETEONCLOSE    = 0x00000008;  /* VFS only */
	private static final int SQLITE_OPEN_EXCLUSIVE        = 0x00000010;  /* VFS only */
	private static final int SQLITE_OPEN_AUTOPROXY        = 0x00000020;  /* VFS only */
	private static final int SQLITE_OPEN_URI              = 0x00000040;  /* Ok for sqlite3_open_v2() */
	private static final int SQLITE_OPEN_MEMORY           = 0x00000080;  /* Ok for sqlite3_open_v2() */
	private static final int SQLITE_OPEN_MAIN_DB          = 0x00000100;  /* VFS only */
	private static final int SQLITE_OPEN_TEMP_DB          = 0x00000200;  /* VFS only */
	private static final int SQLITE_OPEN_TRANSIENT_DB     = 0x00000400;  /* VFS only */
	private static final int SQLITE_OPEN_MAIN_JOURNAL     = 0x00000800;  /* VFS only */
	private static final int SQLITE_OPEN_TEMP_JOURNAL     = 0x00001000;  /* VFS only */
	private static final int SQLITE_OPEN_SUBJOURNAL       = 0x00002000;  /* VFS only */
	private static final int SQLITE_OPEN_MASTER_JOURNAL   = 0x00004000;  /* VFS only */
	private static final int SQLITE_OPEN_NOMUTEX          = 0x00008000;  /* Ok for sqlite3_open_v2() */
	private static final int SQLITE_OPEN_FULLMUTEX        = 0x00010000;  /* Ok for sqlite3_open_v2() */
	private static final int SQLITE_OPEN_SHAREDCACHE      = 0x00020000;  /* Ok for sqlite3_open_v2() */
	private static final int SQLITE_OPEN_PRIVATECACHE     = 0x00040000;  /* Ok for sqlite3_open_v2() */
	private static final int SQLITE_OPEN_WAL              = 0x00080000;  /* VFS only */

	public static final int SQLITE_OK           = 0;   /* Successful result */
/* beginning-of-error-codes */
	public static final int SQLITE_ERROR        = 1;   /* SQL error or missing database */
	public static final int SQLITE_INTERNAL     = 2;   /* Internal logic error in SQLite */
	public static final int SQLITE_PERM         = 3;   /* Access permission denied */
	public static final int SQLITE_ABORT        = 4;   /* Callback routine requested an abort */
	public static final int SQLITE_BUSY         = 5;   /* The database file is locked */
	public static final int SQLITE_LOCKED       = 6;   /* A table in the database is locked */
	public static final int SQLITE_NOMEM        = 7;   /* A malloc() failed */
	public static final int SQLITE_READONLY     = 8;   /* Attempt to write a readonly database */
	public static final int SQLITE_INTERRUPT    = 9;   /* Operation terminated by sqlite3_interrupt()*/
	public static final int SQLITE_IOERR       = 10;   /* Some kind of disk I/O error occurred */
	public static final int SQLITE_CORRUPT     = 11;   /* The database disk image is malformed */
	public static final int SQLITE_NOTFOUND    = 12;   /* Unknown opcode in sqlite3_file_control() */
	public static final int SQLITE_FULL        = 13;   /* Insertion failed because database is full */
	public static final int SQLITE_CANTOPEN    = 14;   /* Unable to open the database file */
	public static final int SQLITE_PROTOCOL    = 15;   /* Database lock protocol error */
	public static final int SQLITE_EMPTY       = 16;   /* Database is empty */
	public static final int SQLITE_SCHEMA      = 17;   /* The database schema changed */
	public static final int SQLITE_TOOBIG      = 18;   /* String or BLOB exceeds size limit */
	public static final int SQLITE_CONSTRAINT  = 19;   /* Abort due to constraint violation */
	public static final int SQLITE_MISMATCH    = 20;   /* Data type mismatch */
	public static final int SQLITE_MISUSE      = 21;   /* Library used incorrectly */
	public static final int SQLITE_NOLFS       = 22;   /* Uses OS features not supported on host */
	public static final int SQLITE_AUTH        = 23;   /* Authorization denied */
	public static final int SQLITE_FORMAT      = 24;   /* Auxiliary database format error */
	public static final int SQLITE_RANGE       = 25;   /* 2nd parameter to sqlite3_bind out of range */
	public static final int SQLITE_NOTADB      = 26;   /* File opened that is not a database file */
	public static final int SQLITE_NOTICE      = 27;   /* Notifications from sqlite3_log() */
	public static final int SQLITE_WARNING     = 28;   /* Warnings from sqlite3_log() */
	public static final int SQLITE_ROW         = 100;  /* sqlite3_step() has another row ready */
	public static final int SQLITE_DONE        = 101;  /* sqlite3_step() has finished executing */

	public static final int SQLITE_INTEGER   = 1;
	public static final int SQLITE_FLOAT     = 2;
	public static final int SQLITE_BLOB      = 4;
	public static final int SQLITE_NULL      = 5;
	public static final int SQLITE_TEXT      = 3;

	public SQLiteDatabase _db = null;

	public String LastErr = "";
	public String _fileName = "";
	public ZumeroDatabase() {
	}
	public static int HandleException(ZumeroDatabase db, Exception e)
	{
		if (db != null)
			db.LastErr = e.toString();
		Log.e("ZUMERO", "exception", e);
		if (e instanceof android.database.sqlite.SQLiteConstraintException)
			return SQLITE_CONSTRAINT;
		if (e instanceof android.database.sqlite.SQLiteAbortException)
			return SQLITE_ABORT;
		if (e instanceof android.database.sqlite.SQLiteAccessPermException)
			return SQLITE_PERM;
		if (e instanceof android.database.sqlite.SQLiteBindOrColumnIndexOutOfRangeException)
			return SQLITE_RANGE;
		if (e instanceof android.database.sqlite.SQLiteBlobTooBigException)
			return SQLITE_TOOBIG;
		if (e instanceof android.database.sqlite.SQLiteCantOpenDatabaseException)
			return SQLITE_CANTOPEN;
		if (e instanceof android.database.sqlite.SQLiteDatabaseCorruptException)
			return SQLITE_CORRUPT;
		if (e instanceof android.database.sqlite.SQLiteDatabaseLockedException)
			return SQLITE_BUSY;
		if (e instanceof android.database.sqlite.SQLiteDatatypeMismatchException)
			return SQLITE_MISMATCH;
		if (e instanceof android.database.sqlite.SQLiteDiskIOException)
			return SQLITE_IOERR;
		if (e instanceof android.database.sqlite.SQLiteDoneException)
			return SQLITE_ERROR;
		if (e instanceof android.database.sqlite.SQLiteFullException)
			return SQLITE_FULL;
		if (e instanceof android.database.sqlite.SQLiteMisuseException)
			return SQLITE_MISUSE;
		if (e instanceof android.database.sqlite.SQLiteOutOfMemoryException)
			return SQLITE_NOMEM;
		if (e instanceof android.database.sqlite.SQLiteReadOnlyDatabaseException)
			return SQLITE_READONLY;
		if (e instanceof android.database.sqlite.SQLiteTableLockedException)
			return SQLITE_LOCKED;
		return 1;
	}
	public static int open_v2(String fileName, ZumeroDatabaseHolder outputHolder, int flags) throws Exception {
		try {
			int javaflags = 0;
			if ((flags & SQLITE_OPEN_CREATE) != 0)
				javaflags |= SQLiteDatabase.CREATE_IF_NECESSARY;
			if ((flags & SQLITE_OPEN_READWRITE) != 0)
				javaflags |= SQLiteDatabase.OPEN_READWRITE;
			if ((flags & SQLITE_OPEN_READONLY) != 0)
				javaflags |= SQLiteDatabase.OPEN_READWRITE;
				//javaflags |= SQLiteDatabase.OPEN_READONLY;
			if ((flags & SQLITE_OPEN_WAL) != 0)
				javaflags |= SQLiteDatabase.ENABLE_WRITE_AHEAD_LOGGING;
			//Log.v("ZUMERO", "java open_v2 " + fileName + " flags " + flags + " javaflags " + javaflags);
			outputHolder.db = new ZumeroDatabase();
			SQLiteDatabase sqdb = SQLiteDatabase.openDatabase(fileName, null, javaflags);
			outputHolder.db._db = sqdb;
			outputHolder.db._fileName = fileName;
			return 0;
		}
		catch (Exception e) {
			return HandleException(outputHolder.db, e);
		}
	}

	public static int prepare_v2(ZumeroDatabase db, String sql, ZumeroStatementHolder outputHolder) throws Exception {
		try {
			outputHolder.stmt = new ZumeroStatement(db, sql);
		}
		catch (Exception e) {
			return HandleException(db, e);
		}
		return 0;
	}

	public static int close(ZumeroDatabase db) throws Exception {
		try {
			if (db != null && db._db != null)
				db._db.close();
		}
		catch (Exception e) {
			return HandleException(db, e);
		}	
		return 0;
	}

	public static int changes(ZumeroDatabase db) throws Exception {
		try {
			SQLiteStatement stmt = db._db.compileStatement("SELECT changes();");
			long val = stmt.simpleQueryForLong();
			stmt.close();
			return (int)val;
		}
		catch (Exception e) {
			HandleException(db, e);
			return 0;
		}
	}

	public static long last_insert_rowid(ZumeroDatabase db) throws Exception {
		try {
			SQLiteStatement stmt = db._db.compileStatement("SELECT last_insert_rowid();");
			long val = stmt.simpleQueryForLong();
			stmt.close();
			return val;
		}
		catch (Exception e) {
			HandleException(db, e);
			return 0;
		}
	}

	public static String errmsg(ZumeroDatabase db) throws Exception {
		return db.LastErr;
	}

	public static int transfer_value(ZumeroStatement stmt_from, ZumeroStatement stmt_to, int colNum_from, int colNum_to)
	{
		int i = colNum_from;
		int rc = -1;
		int colType = ZumeroStatement.column_type(stmt_from, i);
		//Log.v("ZUMERO", "transfer_value colType " + colType);
		switch (colType) {
			case ZumeroDatabase.SQLITE_BLOB:
				{
				ByteBuffer val = ZumeroStatement.column_blob(stmt_from, i);
				if (val == null)
				{
					//Log.v("ZUMERO", "transfer_value binding empty blob");
					rc = ZumeroStatement.bind_blob(stmt_to, colNum_to, ByteBuffer.allocate(0));
				}
				else
					//if (val.array().length > 2000)
						//Log.v("ZUMERO", "binding blob with length " + val.array().length);
					rc = ZumeroStatement.bind_blob(stmt_to, colNum_to, val);
				}
				break;
			case ZumeroDatabase.SQLITE_FLOAT:
				{
				double val = ZumeroStatement.column_double(stmt_from, i);
					//Log.v("ZUMERO", "transfer_value double " + val);
				rc = ZumeroStatement.bind_double(stmt_to, colNum_to, val);
				}
				break;
			case ZumeroDatabase.SQLITE_INTEGER:
				{
				long val = ZumeroStatement.column_int64(stmt_from, i);
					//Log.v("ZUMERO", "transfer_value int " + val);
				rc = ZumeroStatement.bind_int64(stmt_to, colNum_to, val);
				}
				break;
			case ZumeroDatabase.SQLITE_TEXT:
				{
				String val = ZumeroStatement.column_text(stmt_from, i);
					//Log.v("ZUMERO", "transfer_value string " + val);
				rc = ZumeroStatement.bind_text(stmt_to, colNum_to, val);
				}
				break;
			case ZumeroDatabase.SQLITE_NULL:
				{
					//Log.v("ZUMERO", "transfer_value null ");
				rc = ZumeroStatement.bind_null(stmt_to, colNum_to);
				}
				break;
		}
		return rc;
	}

	private static boolean int_array_contains_int(int[] array, int search)
	{
		for (int i = 0; i < array.length; i++)
		{
			if (array[i] == search)
			{
				return true;
			}
		}
		return false;
	}
	private static class ColumnStreamer
	{
		private ArrayList<SQLiteStatement> preparedStatements = new ArrayList<SQLiteStatement>();
		private boolean _isStringColumn = false;
		private byte[] allBytes = null;
		private int bytesAtATime = 1024 * 800; //800K The limit for a cursor is actually somewhere like 1000K, so this is trying to be safe.
		private SQLiteDatabase _db = null;
		private String _origQuery = null;
		private String _column = null;
		private String _querycolumn = null;
		private SQLiteStatement _lengthStatement = null;

		public ColumnStreamer(SQLiteDatabase db, String query, String column, boolean isString)
		{
			_isStringColumn = isString;
			_db = db;
			_origQuery = query;
			_column = column;
			_querycolumn = column;
			if (_column.startsWith("--zss--"))
				_querycolumn = _column.replace("--zss--", "");
			else
				_querycolumn = "[" + _column + "]";
		}
		private void resetAllStatements()
		{
			for (int i = 0; i < preparedStatements.size(); i++)
			{
				preparedStatements.get(i).clearBindings();
			}
			if (_lengthStatement != null)
				_lengthStatement.clearBindings();
		}
		private SQLiteStatement getLengthStatement()
		{
			if (_lengthStatement != null)
				return _lengthStatement;
			else
			{
				String modifiedQuery = _origQuery.replace("{{{ZUMERO_COLUMN_TOKEN}}}", "IFNULL(LENGTH( " + _querycolumn + " ), -1) "); 
				_lengthStatement = _db.compileStatement(modifiedQuery);
				return _lengthStatement;
			}
		}
		private SQLiteStatement getStatementForChunk(int chunkNum)
		{
			if (chunkNum < preparedStatements.size())
				return preparedStatements.get(chunkNum);
			else
			{
				String modifiedQuery;
				if (_isStringColumn)
				{
					modifiedQuery = _origQuery.replace("{{{ZUMERO_COLUMN_TOKEN}}}", "substr(" + _querycolumn + ", " + ((bytesAtATime * chunkNum) + 1) + ", " + bytesAtATime + ")"); 
				}
				else
				{
					modifiedQuery = _origQuery.replace("{{{ZUMERO_COLUMN_TOKEN}}}", _querycolumn);
				}
				SQLiteStatement stmt = _db.compileStatement(modifiedQuery);
				preparedStatements.add(stmt);
                	//Log.v("ZUMERO", "loop_stmt_to_stmt_import_exprt: query for chunkNum " + chunkNum + " is " + modifiedQuery);
				return stmt;
			}
		}
		public String getFullString(int rowid)
		{
			StringBuilder fullString = new StringBuilder();
			int iCurrentChunk = 0;
			int bytesRead = 0;
			while (true)
			{
				SQLiteStatement statement = getStatementForChunk(iCurrentChunk);
				statement.bindLong(1,  rowid);
				String result = statement.simpleQueryForString();
				fullString.append(result);
				if (result == null)
					break;
				bytesRead = result.length();
				iCurrentChunk++;
                	//Log.v("ZUMERO", "loop_stmt_to_stmt_import_exprt: bytesRead " + bytesRead + " bytesAtATime " + bytesAtATime);
				if (bytesRead != bytesAtATime)
					break;
			}
			resetAllStatements();
                	//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid: getFullString" + fullString.length());
			return fullString.toString();
		}
		public byte[] getFullBlob(int rowid)
		{
			SQLiteStatement statementLength = getLengthStatement();
			statementLength.bindLong(1,  rowid);
			long length = statementLength.simpleQueryForLong();
			if (length < 0)
			{
                		//Log.v("ZUMERO", "blob is null");
				return null;
			}
			if (length == 0)
			{
                		//Log.v("ZUMERO", "length is 0");
				return new byte[0];
			}
			SQLiteStatement statement = getStatementForChunk(0);
			statement.bindLong(1,  rowid);
			ParcelFileDescriptor pfd = statement.simpleQueryForBlobFileDescriptor();
			if (pfd == null)
			{
                		//Log.v("ZUMERO", "pfd is null");
				return null;
			}
			FileInputStream stream = new FileInputStream(pfd.getFileDescriptor());
			byte blob[] = new byte[bytesAtATime];
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			try {
				int rc = -1;
				while ((rc = stream.read(blob)) >= 0)
				{
					baos.write(blob, 0, rc);
				}
			}
			catch (IOException io) {
                		//Log.v("ZUMERO", "ioexception");
				resetAllStatements();
				return null;
			}
			resetAllStatements();
                	//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid: getFullBlob size " + baos.size());
			return baos.toByteArray();
		}
		public void closeStatements()
		{
			for (int i = 0; i < preparedStatements.size(); i++)
			{
				preparedStatements.get(i).close();
			}
		}
	}
	public static void loop_stmt_to_stmt_import_export(ZumeroDatabase db_from, ZumeroDatabase db_to, String SelectQueryRowid, String SelectQueryData,
						   String[] columns, int[] largeStringColumnIndexes, int[] largeBlobColumnIndexes,
						   String InsertStatementQuery,
                                                   ZumeroLongHolder rc_from, ZumeroLongHolder rc_to) throws Exception {

               //Log.v("ZUMERO", "loop_stmt_to_stmt_import_export from " + db_from._fileName + " to " + db_to._fileName);
               //Log.v("ZUMERO", "loop_stmt_to_stmt_import_export columns has " + columns.length + " entries. largeStringColumnIndexes has " + largeStringColumnIndexes.length + " entries largeBlobColumnIndexes " + largeBlobColumnIndexes.length + " entries");
               //Log.v("ZUMERO", "loop_stmt_to_stmt_import_export rowid query " + SelectQueryRowid);
               //Log.v("ZUMERO", "loop_stmt_to_stmt_import_export insert query " + InsertStatementQuery);
               int rc = -1;

               int[] colTypes = null;

		Dictionary<String, ColumnStreamer> streamers = new Hashtable<String, ColumnStreamer>();
		ArrayList<String> regularColumns = new ArrayList<String>();
		//The selectQueryRowid looks something like this:
		//"SELECT u._rowid_\n"
            	//"FROM \"z$old$%s\" u INNER JOIN \"z$rv$%s\" rv ON u.z_rv=rv.z_rv  WHERE rv.txid <= ?\n",
		//It doesn't need any updates.

		//The SelectQueryData looks like this:
		//"SELECT {{{ZUMERO_COLUMN_TOKEN}}}\n"
            	//"FROM \"z$old$%s\" u INNER JOIN \"z$rv$%s\" rv ON u.z_rv=rv.z_rv  WHERE u._rowid_= ?\n"
		for (int i = 0; i < columns.length; i++)
		{
			if (int_array_contains_int(largeStringColumnIndexes, i))
			{
				streamers.put(columns[i], new ColumnStreamer(db_from._db, SelectQueryData, columns[i], true));
			}
			else if (int_array_contains_int(largeBlobColumnIndexes, i))
			{
				streamers.put(columns[i], new ColumnStreamer(db_from._db, SelectQueryData, columns[i], false));
			}
			else
			{
				regularColumns.add(columns[i]);
			}
			
		}

		//Prepare the rowid query
		Cursor c_rowid = db_from._db.rawQuery(SelectQueryRowid, new String[] { });
		c_rowid.moveToFirst();

		StringBuilder b = new StringBuilder();
		for(int i = 0; i < regularColumns.size(); i++)
		{
			if (i > 0)
				b.append(",");
			if (regularColumns.get(i).startsWith("--zss--"))
				b.append(regularColumns.get(i).replace("--zss--", ""));
			else
				b.append("[" + regularColumns.get(i) + "]");
		}
		String regColumnsString = b.toString();

		String modifiedQuery = SelectQueryData.replace("{{{ZUMERO_COLUMN_TOKEN}}}", regColumnsString);
		ZumeroStatementHolder stmt_from_holder= new ZumeroStatementHolder();
		rc = ZumeroDatabase.prepare_v2(db_from, modifiedQuery, stmt_from_holder);
		ZumeroStatement stmt_from = stmt_from_holder.stmt;
                if (rc != SQLITE_OK) { rc_from.val = rc; return; }
               //Log.v("ZUMERO", "loop_stmt_to_stmt_import_export data query " + modifiedQuery);

		ZumeroStatementHolder stmt_to_holder= new ZumeroStatementHolder();
		rc = ZumeroDatabase.prepare_v2(db_to, InsertStatementQuery, stmt_to_holder);
		ZumeroStatement stmt_to = stmt_to_holder.stmt;
                if (rc != SQLITE_OK) { rc_to.val = rc; return; }
		
                //Log.v("ZUMERO", "loop_stmt_to_stmt_rowid: number of rows " + c_rowid.getCount());

		while (c_rowid.isAfterLast() == false) {
            		int currentRowid = c_rowid.getInt(0);
			
			//Query all of the regular columns
			
                       //Log.v("ZUMERO", "loop_stmt_to_stmt_rowid: rowid " + currentRowid);
                       ZumeroStatement.bind_int64(stmt_from, 1, currentRowid);
			
                       while ((rc = ZumeroStatement.step(stmt_from)) == ZumeroDatabase.SQLITE_ROW)
                       {
                       		//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid select stepped  " + rc);
			       int regularColumnIndex = 0;
                               rc = ZumeroStatement.clear_bindings(stmt_to);
                               if (rc != SQLITE_OK) { rc_to.val = rc; return; }
                       		//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid clear bindings " + rc);
				int regColumnIndex = 0;
			       for(int i = 0; i < columns.length; i++)
			       {
                       			//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid -- Transferring column " + columns[i]);
				       if (int_array_contains_int(largeStringColumnIndexes, i))
				       {
                       			//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid string column");
					       String longString = streamers.get(columns[i]).getFullString(currentRowid);
					       int rcsub = ZumeroStatement.bind_text(stmt_to, i+1, longString);
                                               if (rcsub != SQLITE_OK) { rc_to.val = rcsub; return; }
				       }
				       else if (int_array_contains_int(largeBlobColumnIndexes, i))
				       {
                       				//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid blob column " + SelectQueryData);
					       byte[] longBlob = streamers.get(columns[i]).getFullBlob(currentRowid);
						//if (longBlob != null)
                       					//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid long blob has " + longBlob.length);
						//else
                       					//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid long blob is null");
					       int rcsub = ZumeroStatement.bind_blob(stmt_to, i+1, longBlob);
                                               if (rcsub != SQLITE_OK) { rc_to.val = rcsub; return; }
				       }
				       else
				       {
                       			//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid transferring value " + regularColumns.get(regularColumnIndex));
                                               int rcsub = transfer_value(stmt_from, stmt_to, regularColumnIndex, i+1);
						regularColumnIndex++;
                                               if (rcsub != SQLITE_OK) { rc_to.val = rcsub; return; }
				       }
			       }
                       	//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid outside stepping stmt_to");
                               rc = ZumeroStatement.step(stmt_to);
                       	//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid outside stepping stmt_to " + rc);
                               if (rc != SQLITE_DONE) { rc_to.val = rc; return; }
                               rc = ZumeroStatement.reset(stmt_to);
                               if (rc != SQLITE_OK) { rc_to.val = rc; return; }
			}
                       	//Log.v("ZUMERO", "loop_stmt_to_stmt_rowid outside loop rc  " + rc);
                       if ( (rc != SQLITE_DONE) ) { rc_from.val = rc; }
                       rc = ZumeroStatement.reset(stmt_from);
                       if (rc != SQLITE_OK) { rc_from.val = rc; return; }
			
       	    		c_rowid.moveToNext();
        	}
        	c_rowid.close();
		ZumeroStatement.finalize(stmt_from);
		ZumeroStatement.finalize(stmt_to);
       }

	public static void loop_stmt_to_stmt(ZumeroStatement stmt_from, ZumeroStatement stmt_to, ZumeroLongHolder rc_from, ZumeroLongHolder rc_to) throws Exception {

		//Log.v("ZUMERO", "loop_stmt_to_stmt between " + stmt_from._db._fileName + " and " + stmt_to._db._fileName );
		int rc = -1;

		int[] colTypes = null;

		int count_columns = -1;
		int j = 0;
		while ((rc = ZumeroStatement.step(stmt_from)) == ZumeroDatabase.SQLITE_ROW)
		{
			//Log.v("ZUMERO", "loop_stmt_to_stmt for row " + j);
			j++;
			if (count_columns < 0)
			{
				count_columns = ZumeroStatement.column_count(stmt_from);
				//colTypes = new int[count_columns];
				//for (int ct_index = 0; ct_index < count_columns; ct_index++)
				//{
					//colTypes[ct_index] = ZumeroStatement.column_type(stmt_from, ct_index);
				//}
			}
			int i=0;
                	rc = ZumeroStatement.clear_bindings(stmt_to);
                	if (rc != SQLITE_OK) { rc_to.val = rc; return; }
                	for (i=0; i<count_columns; ++i)
                	{
				//Log.v("ZUMERO", "loop_stmt_to_stmt for row " + j + " column " + i);
				int rcsub = transfer_value(stmt_from, stmt_to, i, i+1);
				//Log.v("ZUMERO", "transfer_value returned: " + rcsub);
                		if (rcsub != SQLITE_OK) { rc_to.val = rcsub; return; }
                	}
                	rc = ZumeroStatement.step(stmt_to);
			//Log.v("ZUMERO", "step stmt_to returned: " + rc);
                	if (rc != SQLITE_DONE) { rc_to.val = rc; return; }
                	rc = ZumeroStatement.reset(stmt_to);
			//Log.v("ZUMERO", "reset stmt_to returned: " + rc);
                	if (rc != SQLITE_OK) { rc_to.val = rc; return; }
        	}
        	if ( (rc != SQLITE_DONE) ) { rc_from.val = rc; }
	}

public static int stmt_to_stmt(ZumeroStatement stmt_from, int beginFrom, ZumeroStatement stmt_to, int beginTo, int count)
{
	//Log.v("ZUMERO", "stmt_to_stmt between " + stmt_from._db._fileName + " and " + stmt_to._db._fileName );
        int i=0;
	int count_columns = ZumeroStatement.column_count(stmt_from);
	//int[] colTypes = new int[count_columns];
	//for (int ct_index = 0; ct_index < count_columns; ct_index++)
	//{
		//colTypes[ct_index] = ZumeroStatement.column_type(stmt_from, ct_index);
	//}
        if (count < 0) { count = count_columns;  }
        for (i=0; i<count; ++i)
        {
		int rcsub = transfer_value(stmt_from, stmt_to, i + beginFrom, i+1+beginTo);
                if (rcsub != SQLITE_OK) { return rcsub; }
        }

        return SQLITE_OK;
}

public static int fetch_blob_into_file(ZumeroDatabase db, String path, String table, String column, int rowid)
{
	//Log.v("ZUMERO", "fetch_blob_into_file");
	try
	{
		Cursor c = db._db.rawQuery("SELECT [" + column + "] FROM [" + table + "] where _rowid_ = ?", new String[] { Integer.toString(rowid) });
		if (c.moveToFirst())
		{
			byte[] readbuff = new byte[3000];
			byte[] bytes = c.getBlob(0);
			File f = new File(path);
                	f.createNewFile();
			BufferedOutputStream bos = new BufferedOutputStream(
                                               	new FileOutputStream(f));
			bos.write(bytes, 0, bytes.length);
                	bos.flush();
                	bos.close();
		}
	}
	catch (Exception e) {
		HandleException(db, e);
		return 0;
	}
	return ZumeroDatabase.SQLITE_OK;
}

public static int insert_blob_from_file(ZumeroDatabase db, String path, String table, String column, int rowid)
{
	//Log.v("ZUMERO", "insert_blob_from_file");
	try
	{
		File f = new File(path);
		byte[] bytes = new byte[(int)f.length()];

		BufferedInputStream buf = new BufferedInputStream(new FileInputStream(f));
    		buf.read(bytes, 0, bytes.length);
    		buf.close();

		SQLiteStatement stmt = db._db.compileStatement("UPDATE [" + table + "] SET [" + column + "] = ? WHERE _rowid_ = ?");
		stmt.bindBlob(1, bytes);
		stmt.bindLong(2, rowid);
		stmt.execute();
		//stmt.resetAllBindings();
		stmt.close();
	}
	catch (Exception e) {
		HandleException(db, e);
		return 0;
	}
	return ZumeroDatabase.SQLITE_OK;
}
}
