/*
** Copyright 2013-2026 Zumero, LLC
** All rights reserved. 
 */

// IMPORTANT!!!! Do not change the package name, as this
// class is referenced by full package path from JNI.
package com.zumero;

import android.content.Context;

/**
 * A class which holds the various static Zumero methods.<br><br>
 * 
 * <b>Understanding Auth Schemes</b>:  When authenticating requests to the Zumero server,
 * one of the necessary parameters is authScheme.  This is a JSON string, which will
 * be the same throughout your app.  An example auth scheme is <i>{"scheme_type":"default"}</i>". 
 * For more on constructing an auth scheme, see the main Zumero documentation. 
 */
public class ZumeroClient {

	/**
	 * Sync the local database with the server database.
	 * 
	 * @param context An {@link android.content.Context} object.  This must not be null.
	 * @param databasePath The full path to the database file on the device that Zumero should update  This must not be null.
	 * @param encryptionKey The encryption key that should be used when opening the database.  This may be null.
	 * @param serverURL The URL to the Zumero server.  This must not be null.
	 * @param dbFile The name of the DBFile on the Zumero server.  This must not be null.
	 * @param authScheme The Auth Scheme used by the Zumero server to validate the provided credentials.  For help constructing an auth scheme, see {@link com.zumero.ZumeroClient }.  If this is null, the request will be anonymous.
	 * @param username The username to be validated by the Zumero server.  If this is null, the request will be anonymous.
	 * @param password The password to be validated by the Zumero server.  If this is null, the request will be anonymous.
	 */
	public static void sync(Context context, String databasePath, String encryptionKey, String serverURL, String dbFile, String authScheme, String username, String password)
	{
		sync(context, databasePath, encryptionKey, serverURL, dbFile, authScheme, username, password, null);
	}

	/**
	 * Sync the local database with the server database.
	 * 
	 * @param context An {@link android.content.Context} object.  This must not be null.
	 * @param databasePath The full path to the database file on the device that Zumero should update  This must not be null.
	 * @param encryptionKey The encryption key that should be used when opening the database.  This may be null.
	 * @param serverURL The URL to the Zumero server.  This must not be null.
	 * @param dbFile The name of the DBFile on the Zumero server.  This must not be null.
	 * @param authScheme The Auth Scheme used by the Zumero server to validate the provided credentials.  For help constructing an auth scheme, see {@link com.zumero.ZumeroClient }.  If this is null, the request will be anonymous.
	 * @param username The username to be validated by the Zumero server.  If this is null, the request will be anonymous.
	 * @param password The password to be validated by the Zumero server.  If this is null, the request will be anonymous.
	 * @param progListener A progress listener which will be called at various points in the sync process.
	 */
	public static void sync(Context context, String databasePath, String encryptionKey, String serverURL, String dbFile, String authScheme, String username, String password, SyncProgressListener progListener)
	{
		sync(context, databasePath, encryptionKey, serverURL, dbFile, authScheme, username, password, null, null, progListener);
	}

	/**
	 * Sync the local database with the server database.
	 * 
	 * @param context An {@link android.content.Context} object.  This must not be null.
	 * @param databasePath The full path to the database file on the device that Zumero should update  This must not be null.
	 * @param encryptionKey The encryption key that should be used when opening the database.  This may be null.
	 * @param serverURL The URL to the Zumero server.  This must not be null.
	 * @param dbFile The name of the DBFile on the Zumero server.  This must not be null.
	 * @param authScheme The Auth Scheme used by the Zumero server to validate the provided credentials.  For help constructing an auth scheme, see {@link com.zumero.ZumeroClient }.  If this is null, the request will be anonymous.
	 * @param username The username to be validated by the Zumero server.  If this is null, the request will be anonymous.
	 * @param password The password to be validated by the Zumero server.  If this is null, the request will be anonymous.
	 * @param jsOptions JSON string requesting special sync operations. If this is null, sync will proceed as normal.
	 * @param details If non-null, the object's syncId member will receive a sync details token (if requested in jsOptions).
	 * @param progListener A progress listener which will be called at various points in the sync process.
	 */
	public static void sync(Context context, String databasePath, String encryptionKey, String serverURL, String dbFile, String authScheme, String username, String password, String jsOptions, SyncDetails details, SyncProgressListener progListener)
	{
		if (context == null)
			throw new NullPointerException("A context must be provided to ZumeroClient.Sync");
		if (databasePath == null)
			throw new NullPointerException("A databasePath must be provided to ZumeroClient.Sync");
		if (serverURL == null)
			throw new NullPointerException("A serverURL must be provided to ZumeroClient.Sync");
		if (dbFile == null)
			throw new NullPointerException("A dbFile must be provided to ZumeroClient.Sync");
		native_sync3(databasePath, encryptionKey, serverURL, dbFile, authScheme, username, password, context.getCacheDir().getAbsolutePath(), jsOptions, details, progListener);
	}

	/**
	 * Move un-synced local changes into an isolated holding area. Typically, the
	 * reason to do so is because the local changes conflict with other changes
	 * already on the server.
	 *
	 * The quarantined changes are stored locally, so no network activity is
	 * required.
	 * 
	 * @param context An {@link android.content.Context} object.  This must not be null.
	 * @param databasePath The full path to the database file on the device that Zumero should update  This must not be null.
	 * @param encryptionKey The encryption key that should be used when opening the database.  This may be null.
	 * @param jsOptions JSON string requesting special sync operations. If this is null, the operation will proceed as normal.
	 * @return The quarantineID, which can be used to sync or delete the quarantine later.
	 */
	public static long quarantineSinceLastSync(Context context, String databasePath, String encryptionKey, String jsOptions)
	{
		if (context == null)
			throw new NullPointerException("A context must be provided to ZumeroClient.QuarantineSinceLastSync");
		if (databasePath == null)
			throw new NullPointerException("A databasePath must be provided to ZumeroClient.QuarantineSinceLastSync");
		return native_quarantine_since_last_sync3(databasePath, encryptionKey, context.getCacheDir().getAbsolutePath(), jsOptions);
	}

	/**
	 * Move un-synced local changes into an isolated holding area. Typically, the
	 * reason to do so is because the local changes conflict with other changes
	 * already on the server.
	 *
	 * The quarantined changes are stored locally, so no network activity is
	 * required.
	 * 
	 * @param context An {@link android.content.Context} object.  This must not be null.
	 * @param databasePath The full path to the database file on the device that Zumero should update  This must not be null.
	 * @param encryptionKey The encryption key that should be used when opening the database.  This may be null.
	 * @return The quarantineID, which can be used to sync or delete the quarantine later.
	 */
	public static long quarantineSinceLastSync(Context context, String databasePath, String encryptionKey)
	{
		return quarantineSinceLastSync(context, databasePath, encryptionKey, null);
	}
	
	/**
	 * Sync the local database with the server database, including the changes
	 * stored in the specified quarantine.
	 * 
	 * @param context An {@link android.content.Context} object.  This must not be null.
	 * @param databasePath The full path to the database file on the device that Zumero should update  This must not be null.
	 * @param encryptionKey The encryption key that should be used when opening the database.  This may be null.
	 * @param quarantineID The quarantineID returned from  {@link #quarantineSinceLastSync(Context, String, String)}
	 * @param serverURL The URL to the Zumero server.  This must not be null.
	 * @param dbFile The name of the DBFile on the Zumero server.  This must not be null.
	 * @param authScheme The Auth Scheme used by the Zumero server to validate the provided credentials.  For help constructing an auth scheme, see {@link com.zumero.ZumeroClient }.  If this is null, the request will be anonymous.
	 * @param username The username to be validated by the Zumero server.  If this is null, the request will be anonymous.
	 * @param password The password to be validated by the Zumero server.  If this is null, the request will be anonymous.
	 */
	public static void syncQuarantine(Context context, String databasePath, String encryptionKey, long quarantineID, String serverURL, String dbFile, String authScheme, String username, String password)
	{
		syncQuarantine(context, databasePath, encryptionKey, quarantineID, serverURL, dbFile, authScheme, username, password, null, null, null);
	}

	/**
	 * Sync the local database with the server database, including the changes
	 * stored in the specified quarantine.
	 * 
	 * @param context An {@link android.content.Context} object.  This must not be null.
	 * @param databasePath The full path to the database file on the device that Zumero should update  This must not be null.
	 * @param encryptionKey The encryption key that should be used when opening the database.  This may be null.
	 * @param quarantineID The quarantineID returned from  {@link #quarantineSinceLastSync(Context, String, String)}
	 * @param serverURL The URL to the Zumero server.  This must not be null.
	 * @param dbFile The name of the DBFile on the Zumero server.  This must not be null.
	 * @param authScheme The Auth Scheme used by the Zumero server to validate the provided credentials.  For help constructing an auth scheme, see {@link com.zumero.ZumeroClient }.  If this is null, the request will be anonymous.
	 * @param username The username to be validated by the Zumero server.  If this is null, the request will be anonymous.
	 * @param password The password to be validated by the Zumero server.  If this is null, the request will be anonymous.
	 * @param jsOptions JSON string requesting special sync operations. If this is null, sync will proceed as normal.
	 * @param details If non-null, the object's syncId member will receive a sync details token (if requested in jsOptions).
	 * @param progListener A progress listener which will be called at various points in the sync process.
	 */
	public static void syncQuarantine(Context context, String databasePath, String encryptionKey, long quarantineID, String serverURL, String dbFile, String authScheme, String username, String password, String jsOptions, SyncDetails details, SyncProgressListener progListener)
	{
		if (context == null)
			throw new NullPointerException("A context must be provided to ZumeroClient.SyncQuarantine");
		if (databasePath == null)
			throw new NullPointerException("A databasePath must be provided to ZumeroClient.SyncQuarantine");
		if (serverURL == null)
			throw new NullPointerException("A serverURL must be provided to ZumeroClient.SyncQuarantine");
		if (dbFile == null)
			throw new NullPointerException("A dbFile must be provided to ZumeroClient.SyncQuarantine");
		native_sync_quarantine3(databasePath, encryptionKey, quarantineID, serverURL, dbFile, authScheme, username, password, context.getCacheDir().getAbsolutePath(), jsOptions, details, progListener);
	}
	
	/**
	 * Permanently delete quarantined changes.
	 * 
	 * @param context An {@link android.content.Context} object.  This must not be null.
	 * @param databasePath The full path to the database file on the device that Zumero should update  This must not be null.
	 * @param encryptionKey The encryption key that should be used when opening the database.  This may be null.  * @param quarantineID The quarantineID returned from  {@link #quarantineSinceLastSync(Context, String, String)}
	 */
	public static void deleteQuarantine(Context context, String databasePath, String encryptionKey, long quarantineID)
	{
		deleteQuarantine(context, databasePath, encryptionKey, quarantineID, null);
	}
	
	/**
	 * Permanently delete quarantined changes.
	 * 
	 * @param context An {@link android.content.Context} object.  This must not be null.
	 * @param databasePath The full path to the database file on the device that Zumero should update  This must not be null.
	 * @param encryptionKey The encryption key that should be used when opening the database.  This may be null.  * @param quarantineID The quarantineID returned from  {@link #quarantineSinceLastSync(Context, String, String)}
	 * @param jsOptions JSON string requesting special sync operations. If this is null, the operation will proceed as normal.
	 */
	public static void deleteQuarantine(Context context, String databasePath, String encryptionKey, long quarantineID, String jsOptions)
	{
		if (context == null)
			throw new NullPointerException("A context must be provided to ZumeroClient.QuarantineSinceLastSync");
		if (databasePath == null)
			throw new NullPointerException("A databasePath must be provided to ZumeroClient.QuarantineSinceLastSync");
		native_delete_quarantine3(databasePath, encryptionKey, quarantineID, jsOptions);
	}
	/**
	 * Cancel a sync that is currently ongoing.
	 * @param cancel_token The token that was provided to the SyncProgressListener's onSyncProgress method.
	 */
	public static void cancel(int cancel_token)
	{
		native_cancel(cancel_token);
	}

	public static class SyncDetails 
	{
		public int syncId;
	}

	native static void native_sync(String databasePath, String encryptionKey, String serverURL, String dbFile, String authScheme, String username, String password, String tmpDir);

	native static void native_sync2(String databasePath, String encryptionKey, String serverURL, String dbFile, String authScheme, String username, String password, String tmpDir, SyncProgressListener listener);
	
	native static void native_sync3(String databasePath, String encryptionKey, String serverURL, String dbFile, String authScheme, String username, String password, String tmpDir, String jsOptions, SyncDetails details, SyncProgressListener listener);
	
	native static long native_quarantine_since_last_sync(String databasePath, String encryptionKey, String tmpDir);
	native static long native_quarantine_since_last_sync3(String databasePath, String encryptionKey, String tmpDir, String jsOptions);
	
	native static void native_sync_quarantine(String databasePath, String encryptionKey, long quarantineID, String serverURL, String dbFile, String authScheme, String username, String password, String tmpDir);
	native static void native_sync_quarantine3(String databasePath, String encryptionKey, long quarantineID, String serverURL, String dbFile, String authScheme, String username, String password, String tmpDir, String jsOptions, SyncDetails details, SyncProgressListener listener);
	
	native static void native_delete_quarantine(String databasePath, String encryptionKey, long quarantineID);
	native static void native_delete_quarantine3(String databasePath, String encryptionKey, long quarantineID, String jsOptions);
	native static void native_cancel(int cancel_token);
	
	 static {
		 System.loadLibrary("zumero_sqlite_functions");
		 System.loadLibrary("zumero_client_api");
		 System.loadLibrary("zumero_jni");
	 }
}
